/* -----------------------------------------------------------------------
 * File: buildin.c
 * Author: Bernhard Holzinger 12.09.96
 * Purpose: This file contains the function prototypes for the buildin
 * test function.
 * Rev 3/98 SCR; 2926 compatibility.  This has been modified extensively
 * and static properties are no longer stored in the Best Memory.
 * The 2925 used static vars for property get/set...now stored here.
 * ----------------------------------------------------------------------- */

#include <typedefs.h>

#include <builtin.h>
#include <dynamic.h>
#include <iocommon.h>
#include <decode.h>
#include <errcapi.h>
#include <ladata.h>
#include <master.h>
#include <observer.h>
#include <regconst.h>
#include <session.h>

/* TODO: remove this as soon as srget has found a good place */
#include <mini_api.h>


/* local defines */
#define BLOCKLENGTH 		1300
#define HOSTBUFFER 		1024
#define MAX_PROT_ERRORS 	24
#define B_TSTCMD_BLOCKMOVE_C 	7

#define LOWER_HALF_ADDR   0x0UL
#define UPPER_HALF_ADDR   0x10000UL


/* a 'safe' way to free (freeptr used to avoid warnings) */
static void * freeptr = NULL;
#define FREE_N_NULL(ptr) free(ptr), freeptr = ptr = NULL;

/* avoid compiler warnings in asserts */
static int Abort = 0;

/* --------------------------------------------------------------------------
 * statics
 * -------------------------------------------------------------------------- */

/******************************************************************************/
/* Disable Mem-Write & Inval when no Cacheline used. */

static int fCanUseMWI = 0;


/******************************************************************************/
/* The B_PAP structure provides constants for the calls to
 * ProgAttrPage() in the BestTestRun() routine.
 *
 * 2925; Each entry describes all the things that are common to a SERIES of
 * Master Block Transactions  (from StartEntryNum to EndEntryNum),
 * all of which point to ONE attribute PAGE and are performed in firmware.
 *
 * 2926; Each entry describes all phases of ONE attribute PAGE.
 * Between 1 and 3 Master Block Entries are used per page, depending on
 * the type of test.
 *
 * Thus (for both 2925 and 2926);
 * Each test uses ONE block page and as many attribute pages as there
 * are elements in the structure for that type of test.
 * One or more block page entries may thus reference the same attribute page.
 *
 */

typedef struct _B_PAP           /* Best-Program-Attribute-Page */
{
  b_int32 StartEntryNum;
  b_int32 EndEntryNum;
  b_int32 BurstFactor;          /* in cache line size increments */
  b_int32 NumWaits;
  b_int32 ReadCmd;
  b_int32 WriteCmd;
} B_PAP;

static const B_PAP s_prot_lite[] =
{
  {0, 7, 1, 0, B_CMD_MEM_READ, B_CMD_MEM_WRITE}
};

static const B_PAP s_prot_med[] =
{
  {0, 1, 0, 0, B_CMD_MEM_READ, B_CMD_MEM_WRITE},
  {2, 3, 0, 1, B_CMD_MEM_READ, B_CMD_MEM_WRITE},
  {4, 5, 1, 0, B_CMD_MEM_READLINE, B_CMD_MEM_WRITE},
  {6, 7, 1, 1, B_CMD_MEM_READLINE, B_CMD_MEM_WRITE}
};

static const B_PAP s_prot_hard[] =
{
  {0, 0, 0, 0, B_CMD_MEM_READ, B_CMD_MEM_WRITE},
  {1, 1, 0, 1, B_CMD_MEM_READ, B_CMD_MEM_WRITE},
  {2, 2, 1, 0, B_CMD_MEM_READLINE, B_CMD_MEM_WRITE},
  {3, 3, 1, 1, B_CMD_MEM_READLINE, B_CMD_MEM_WRITE},
  {4, 4, 1, 2, B_CMD_MEM_READLINE, B_CMD_MEM_WRITEINVALIDATE},
  {5, 5, 2, 0, B_CMD_MEM_READMULTIPLE, B_CMD_MEM_WRITEINVALIDATE},
  {6, 6, 3, 1, B_CMD_MEM_READMULTIPLE, B_CMD_MEM_WRITEINVALIDATE},
  {7, 7, 4, 0, B_CMD_MEM_READMULTIPLE, B_CMD_MEM_WRITEINVALIDATE}
};
#define SIZEOF_S_PROT_LITE  (sizeof(s_prot_lite) / sizeof(B_PAP))
#define SIZEOF_S_PROT_MED   (sizeof(s_prot_med) / sizeof(B_PAP))
#define SIZEOF_S_PROT_HARD  (sizeof(s_prot_hard) / sizeof(B_PAP))




/* --------------------------------------------------------------------------
 * Data structures to handle Properties
 * -------------------------------------------------------------------------- */

typedef struct _BIT_PROPS
{
  b_int8 TestBandwidth;
  b_int8 TestDataPattern;
  b_int8 TestProtocol;
  b_int8 TestCompare;
  b_int32 TestBlkLength;
  b_int32 TestStartAddr;        /* NOT USED !!! */
  b_int32 TestNofBytes;
  b_int32 TestSourceAddr;
  b_int32 TestDestAddr;
  b_int32 TestTimeDist;
  b_int32 TestTimeLimit;

} BIT_PROPS;
/* default properties */
#define DEF_PROPS   {  50, B_DATAPATTERN_RANDOM, B_PROTOCOL_LITE, 0, \
                       8192, 0, 4, 0, 0, 1000, 1000 }

static const BIT_PROPS def_bitprops = DEF_PROPS;

/* current properties */
static BIT_PROPS bit_props[MAXHANDLES + 1] = {
  DEF_PROPS, DEF_PROPS, DEF_PROPS, DEF_PROPS,
  DEF_PROPS, DEF_PROPS, DEF_PROPS, DEF_PROPS,
  DEF_PROPS, DEF_PROPS, DEF_PROPS, DEF_PROPS,
  DEF_PROPS, DEF_PROPS, DEF_PROPS, DEF_PROPS,
  DEF_PROPS, DEF_PROPS, DEF_PROPS, DEF_PROPS,
  DEF_PROPS, DEF_PROPS, DEF_PROPS, DEF_PROPS,
  DEF_PROPS, DEF_PROPS, DEF_PROPS, DEF_PROPS,
  DEF_PROPS, DEF_PROPS, DEF_PROPS, DEF_PROPS,
  DEF_PROPS, DEF_PROPS, DEF_PROPS, DEF_PROPS,
  DEF_PROPS, DEF_PROPS, DEF_PROPS, DEF_PROPS,
  DEF_PROPS, DEF_PROPS, DEF_PROPS, DEF_PROPS,
  DEF_PROPS, DEF_PROPS, DEF_PROPS, DEF_PROPS,
  DEF_PROPS, DEF_PROPS, DEF_PROPS, DEF_PROPS,
  DEF_PROPS, DEF_PROPS, DEF_PROPS, DEF_PROPS,
  DEF_PROPS, DEF_PROPS, DEF_PROPS, DEF_PROPS,
  DEF_PROPS, DEF_PROPS, DEF_PROPS, DEF_PROPS,
  DEF_PROPS, DEF_PROPS, DEF_PROPS, DEF_PROPS,
  DEF_PROPS, DEF_PROPS, DEF_PROPS, DEF_PROPS,
  DEF_PROPS, DEF_PROPS, DEF_PROPS, DEF_PROPS,
  DEF_PROPS, DEF_PROPS, DEF_PROPS, DEF_PROPS,
  DEF_PROPS, DEF_PROPS, DEF_PROPS, DEF_PROPS,
  DEF_PROPS, DEF_PROPS, DEF_PROPS, DEF_PROPS,
  DEF_PROPS, DEF_PROPS, DEF_PROPS, DEF_PROPS,
  DEF_PROPS, DEF_PROPS, DEF_PROPS, DEF_PROPS,
  DEF_PROPS, DEF_PROPS, DEF_PROPS, DEF_PROPS,
  DEF_PROPS, DEF_PROPS, DEF_PROPS, DEF_PROPS,
  DEF_PROPS, DEF_PROPS, DEF_PROPS, DEF_PROPS,
  DEF_PROPS, DEF_PROPS, DEF_PROPS, DEF_PROPS,
  DEF_PROPS, DEF_PROPS, DEF_PROPS, DEF_PROPS,
  DEF_PROPS, DEF_PROPS, DEF_PROPS, DEF_PROPS,
  DEF_PROPS, DEF_PROPS, DEF_PROPS, DEF_PROPS,
  DEF_PROPS, DEF_PROPS, DEF_PROPS, DEF_PROPS, DEF_PROPS
};

#if MAXHANDLES > 128
#error Initialization of bit_props invalidated by increase in MAXHANDLES
#endif


/* --------------------------------------------------------------------------
 * Convert a second-counter into hours:mins:secs
 * -------------------------------------------------------------------------- */

static struct TM_SHORT
{
  int hour;
  int minute;
  int sec;
} hms;

static void ConvertTime(b_int32 timer_value, struct TM_SHORT * pHms)
{
  pHms->hour = (int) (timer_value / 3600);
  pHms->minute = (int) ((timer_value / 60) % 60);
  pHms->sec = (int) (timer_value % 60);
}


/* --------------------------------------------------------------------------
 * Attribute page size varies between 2925 and 2926
 * -------------------------------------------------------------------------- */

#define ATTR_PAGE_SIZE_2925 32
#define ATTR_PAGE_SIZE_2926 4

static b_int32 GetAttrPageSize(b_handletype handle)
{
  return BestIs2925(handle) ? ATTR_PAGE_SIZE_2925 : ATTR_PAGE_SIZE_2926;
}

/* --------------------------------------------------------------------------
 * Errors written to the report file (not part of Best error handling)
 * -------------------------------------------------------------------------- */

typedef enum _ENUM_ERR_TEST
{
  ERR_TEST_PRODNUM,
  ERR_TEST_SERIALNUM,
  ERR_TEST_TIMERVAL,
  ERR_TEST_STATUS,
  ERR_TEST_OBSRESULT,
  ERR_TEST_TRACEWRITE,
  ERR_TEST_MALLOC,
  ERR_TEST_RESULTS_LOAD,
  ERR_TEST_NONE
} ENUM_ERR_TEST;

static void ReportError(ENUM_ERR_TEST error_num, FILE *fileptr)
{
  switch (error_num)
  {
  case ERR_TEST_NONE:
    break;

  case ERR_TEST_PRODNUM:
    fprintf(fileptr, "\n\nERROR while reading product number!");
    break;

  case ERR_TEST_SERIALNUM:
    fprintf(fileptr, "\n\nERROR while reading serial number!");
    break;

  case ERR_TEST_TIMERVAL:
    fprintf(fileptr, "\n\nERROR while reading timer value!");
    break;

  case ERR_TEST_STATUS:
    fprintf(fileptr, "\n\nERROR while reading status information!");
    break;

  case ERR_TEST_OBSRESULT:
    fprintf(fileptr, "\n\nERROR while reading observer results!");
    break;

  case ERR_TEST_TRACEWRITE:
    fprintf(fileptr, "\n\nERROR while writing trace to file!");
    break;

  case ERR_TEST_MALLOC:
    fprintf(fileptr, "\n\nERROR while allocating memory!");
    break;

  case ERR_TEST_RESULTS_LOAD:
    fprintf(fileptr, "\n\nERROR while loading test results!");
    break;

  default:
    fprintf(fileptr, "\n\nUNKNOWN ERROR !");
  }
}


/* --------------------------------------------------------------------------
 * Returns true/false if the test failed (depends on 2925/2926)
 * -------------------------------------------------------------------------- */
#define OBS_STATUS_RUNNING          0x1UL
#define OBS_STATUS_OUTOFSYNC        0x2UL
#define OBS_STATUS_PROTERR          0x4UL

#define LA_STATUS_TRIGGERED         0x2UL

#define STATUS_MASTER_RUNNING       0x1UL
#define STATUS_TARGET_RUNNING       0x2UL

#define STATUS_LA_RUNNING           0x8UL

#define STATUS_DATA_COMPARE_ERR     0x20UL
#define STATUS_MASTER_BLOCK_ABORT   0x80UL
#define STATUS_BUILT_IN_TEST_FAIL   0x1000UL
#define STATUS_ANY_ERROR \
  (STATUS_DATA_COMPARE_ERR | STATUS_MASTER_BLOCK_ABORT | STATUS_BUILT_IN_TEST_FAIL)


static int HasTestFailed(
    b_handletype handle,
    b_int32 best_status_mask)
{
  if(BestIs2925(handle))
  {
    return(0 != (best_status_mask & STATUS_BUILT_IN_TEST_FAIL));
  }
  else
  {
    return(0 != (best_status_mask & 
      (STATUS_DATA_COMPARE_ERR | STATUS_MASTER_BLOCK_ABORT)));
  }
}

/* --------------------------------------------------------------------------
 * Dumps observer error strings into the report file
 * -------------------------------------------------------------------------- */

static void ReportObsErrors(
    b_handletype handle,
    FILE *pRptFile,
    b_int32 error_mask)
{
  unsigned int i;
  b_charptrtype pMsg;

  if ((NULL == pRptFile) || (0 == error_mask))
    return;

  for (i = 0; i < MAX_PROT_ERRORS; i++)
  {
    if (error_mask & (1UL << i))
    {
      if (B_E_OK != BestObsErrStringGet(handle, (b_int32)i, &pMsg))
        return;
      fprintf(pRptFile, "* %s\n", pMsg);
    }
  }
}


/* --------------------------------------------------------------------------
 * Stops all mem. decoders starting at ...
 * -------------------------------------------------------------------------- */

static b_errtype StopAllDecoders(
    b_handletype handle,
    b_decodertype eStartAtDecNum
)
{
  b_int32 DecMode;
  b_int32 DecSize;
  b_decodertype DecNum;
  B_TRY_VARS_NO_PROG;

  B_TRY_BEGIN
  {
    for (DecNum = eStartAtDecNum; DecNum <= B_DEC_NODECODER; DecNum++)
    {
      B_TRY(BestTargetDecoderPropGet(handle, B_DEC_SIZE, &DecSize));

      if (0 != DecSize)
      {
        /* An enabled decoder ... check location */
        B_TRY(BestTargetDecoderPropGet(handle, B_DEC_MODE, &DecMode));

        if (B_MODE_MEM == DecMode)
        {
          /* An enabled decoder in mem space; turn it off */
          B_TRY(BestTargetDecoderPropDefaultSet(handle, DecNum));
          B_TRY(BestTargetDecoderPropSet(handle, B_DEC_SIZE, 0UL));
          B_TRY(BestTargetDecoderPropSet(handle, B_DEC_BASEADDR, 0UL));
          B_TRY(BestTargetDecoderProg(handle, DecNum));
        }
      }
    }
  }

  B_ERRETURN( B_TRY_RET);
}



/* --------------------------------------------------------------------------
 * ProgramBlockPage() can Init a master block page or append an entry to it.
 * -------------------------------------------------------------------------- */

typedef enum _PBP_MODE
{
  BLOCKPAGE_INIT,
  BLOCKPAGE_APPEND
} PBP_MODE;

static b_errtype ProgramBlockPage(
    b_handletype handle,
    PBP_MODE mode,
    b_int32 BusCmd,
    b_int32 BusAddr,
    b_int32 IntAddr,
    b_int32 AttrPageNum,
    b_int32 NumDWords,
    b_int32 CompFlag,
    b_int32 CompOffset
)
{
  B_TRY_VARS_NO_PROG;

  /* it's not an error to call this with a 2925 ... just ignore and return ok */
  if (BestIs2925(handle))
    B_ERRETURN( B_E_OK);

  B_TRY_BEGIN
  {
    switch (mode)
    {
    case BLOCKPAGE_INIT:
      /* stop any test in progress */
      B_TRY(BestMasterStop(handle));

      /* set the master generic properties */
      B_TRY(BestMasterGenPropDefaultSet(handle));

      B_TRY(BestExerciserGenPropSet(handle, B_EGEN_ATTRPAGESIZE, GetAttrPageSize(handle)));
      B_TRY(BestMasterGenPropSet(handle, B_MGEN_RUNMODE,
            (b_int32)B_RUNMODE_IMMEDIATE));
      B_TRY(BestMasterGenPropSet(handle, B_MGEN_ATTRMODE,
            (b_int32)B_ATTRMODE_BLOCK));
      B_TRY(BestMasterGenPropSet(handle, B_MGEN_REPEATMODE,
            (b_int32)B_REPEATMODE_INFINITE));

      B_TRY(BestMasterGenPropSet(handle, B_MGEN_MASTERENABLE, 1UL));
      B_TRY(BestMasterGenPropSet(handle, B_MGEN_MWIENABLE, 1UL));

      /* init the master block page that we'll use */
      B_TRY(BestMasterBlockPageInit(handle, 1UL));
      B_TRY(BestMasterBlockPropDefaultSet(handle));
      break;

    case BLOCKPAGE_APPEND:
      /* Each Master Attr. Page gets at LEAST one Block Page Entry */
      B_TRY(BestMasterBlockPropSet(handle, B_BLK_BUSCMD, BusCmd));
      B_TRY(BestMasterBlockPropSet(handle, B_BLK_BUSADDR, BusAddr));
      B_TRY(BestMasterBlockPropSet(handle, B_BLK_BYTEN, 0UL));
      B_TRY(BestMasterBlockPropSet(handle, B_BLK_INTADDR, IntAddr));
      B_TRY(BestMasterBlockPropSet(handle, B_BLK_NOFDWORDS, NumDWords));
      B_TRY(BestMasterBlockPropSet(handle, B_BLK_ATTRPAGE, AttrPageNum));
      B_TRY(BestMasterBlockPropSet(handle, B_BLK_COMPFLAG, CompFlag));
      B_TRY(BestMasterBlockPropSet(handle, B_BLK_COMPOFFS, CompOffset));

      B_TRY(BestMasterBlockProg(handle));
      break;

    default:
      break;
    }
  }

  B_ERRETURN( B_TRY_RET);
}


/* --------------------------------------------------------------------------
 * AddBlockPageEntries() "decodes" the TestCmd and adds appropriate
 * entries to the master block page to make a complete transaction for ONE
 * attribute page.
 * -------------------------------------------------------------------------- */

static b_errtype AddBlockPageEntries(
    b_handletype handle,
    const B_PAP * pProt,
    b_int32 CmdMask,
    b_int32 BusAddr1,           /* source */
    b_int32 BusAddr2,           /* destination */
    b_int32 AttrPageNum,
    b_int32 NumDWords
)
{
  B_TRY_VARS_NO_PROG;

  b_int32 CompFlag = 0;
  b_int32 CompOffset = 0;
  b_int32 WriteCmd = fCanUseMWI ? pProt->WriteCmd : B_CMD_MEM_WRITE;
  /* it's not an error to call this with a 2925 ... just ignore and return ok */
  if (BestIs2925(handle))
    B_ERRETURN( B_E_OK);

  B_TRY_BEGIN
  {
    /* Possibility 1 */
    if (CmdMask & PBP_READ_TO_UPPER_HALF)
    {
      B_TRY(ProgramBlockPage(handle, BLOCKPAGE_APPEND, pProt->ReadCmd,
          BusAddr1, UPPER_HALF_ADDR, AttrPageNum,
          NumDWords, 0UL, 0UL));
    }

    /* Possibility 2 */
    if (CmdMask & PBP_WRITE_FROM_UPPER_HALF)
    {
      B_TRY(ProgramBlockPage(handle, BLOCKPAGE_APPEND, WriteCmd,
          BusAddr2, UPPER_HALF_ADDR, AttrPageNum,
          NumDWords, 0UL, 0UL));
    }

    /* Possibility 3 */
    if (CmdMask & PBP_READ_TO_LOWER_HALF)
    {
      if (CmdMask & PBP_COMPARE)
      {
        /* compare with data in upper half of int. mem */
        CompFlag = 1;
        CompOffset = UPPER_HALF_ADDR;
      }

      /* NOTE; we read from where we just wrote to for a compare */

      B_TRY(ProgramBlockPage(handle, BLOCKPAGE_APPEND, pProt->ReadCmd,
          BusAddr2, LOWER_HALF_ADDR, AttrPageNum,
          NumDWords, CompFlag, CompOffset));
    }
  }

  B_ERRETURN( B_TRY_RET);

}
/* --------------------------------------------------------------------------
 * ProgAttrPage() adds all phases required for ONE attr. page
 * -------------------------------------------------------------------------- */

static b_errtype ProgAttrPage(
    b_handletype handle,
    const B_PAP * pProt,
    b_int32 AttrPageNum,
    b_int32 CacheLine,
    b_int32 TestCmd,
    b_int32 * pAttrPageLen)
{
  b_int32 AttrPageLen;
  b_int32 NumTransfers;
  b_int32 NumPageRepeats;
  b_int32 PBuffer[5];
  b_int32 i;
  b_int8 SetupCmd;
  b_int32 MinTransfers;
  b_int32 ClksPerAttrPage;
  b_int32 ClksDelayPerAttrPage;

  int fIs2925 = BestIs2925(handle);

  b_int32 WriteCmd = fCanUseMWI ? pProt->WriteCmd : B_CMD_MEM_WRITE;
  B_TRY_VARS_NO_PROG;

  /* calculate # of attr. phases for this page... don't let it be 0 !! */
  AttrPageLen = (pProt->BurstFactor == 0) ? 
    CacheLine / 2 : 
    CacheLine * pProt->BurstFactor;

  AttrPageLen = max(1, AttrPageLen);

  *pAttrPageLen = AttrPageLen;

  /* # of clock cycles needed for this attribute page */
  ClksPerAttrPage = 7 + AttrPageLen + ((AttrPageLen - 1) * pProt->NumWaits);

  /* get correct % of bus bandwidth used (# of times to run each page) 
   * NOTE; BLOCKLENGTH is in clocks 
   */
  NumPageRepeats = ((b_int32)(bit_props[handle].TestBandwidth) * (b_int32)BLOCKLENGTH) 
    / ((b_int32)100 * (b_int32)ClksPerAttrPage);

  /* Total # of dword transfers that we'll make in each duty cycle - 2925 */
  NumTransfers = NumPageRepeats * AttrPageLen;

  switch (TestCmd)
  {
   case B_TSTCMD_TRAFFICMAKE:
   case B_TSTCMD_READ:
    MinTransfers = 8;
    break;

   case B_TSTCMD_WRITEREAD:
    MinTransfers = 16;
    break;

   case B_TSTCMD_BLOCKMOVE:
    MinTransfers = bit_props[handle].TestCompare ? 24 : 16;
    break;

   default:
    MinTransfers = 0;
    assert(Abort && "Unanticipated case (switch (TestCmd) in ProgAttrPage())");
  }

  while (NumTransfers < MinTransfers)
    NumTransfers += AttrPageLen;

  /* adjust NumTransfers to be on a CacheLine boundary */
  if (NumTransfers % CacheLine)
  {
    NumTransfers += (CacheLine - (NumTransfers % CacheLine));
  }

  /* make NumPageRepeats real...on the 2926 this is the number of times we'll
   * repeat each attribute phase.  Note that AttrPageLen is already aligned to
   * the cache line size so we cannot misalign here. 
   */
   
  NumPageRepeats = NumTransfers / AttrPageLen;

  /* 2926 ONLY; # of clock cycles to remain idle between attr. page loops 
   * (ClksPerAttrPage - 7) ignores the initial overhead.
   * Note; use a multiplier of 100 to achieve resolution with integer div.
   */
   
  ClksDelayPerAttrPage = (ClksPerAttrPage - 7) * NumPageRepeats *
    (((10000 / bit_props[handle].TestBandwidth) - 100) / 100);


  B_TRY_BEGIN
    {
      /* init the attribute page */

      B_TRY(BestMasterAttrPageInit(handle, AttrPageNum));

      /* programm all attribute phases */

      for (i = 1; i <= AttrPageLen; i++)
      {
	/* this has to be within the loop...done for each phase */

	B_TRY(BestMasterAttrPropDefaultSet(handle));

	if ((1 == i) && !fIs2925)
	{
	  /* 2926 is programmed differently .... delay BEFORE the first
	   * (address) phase of the attr. page to get idle time. */
	  B_TRY(BestMasterAttrPropSet(handle, B_M_DELAY, ClksDelayPerAttrPage));
	  /* repeat the data phase x times to get duty cycle */
	  B_TRY(BestMasterAttrPropSet(handle, B_M_REPEAT, NumPageRepeats));
	}

	if (i == AttrPageLen)
	{
	  /* last phase */
	  B_TRY(BestMasterAttrPropSet(handle, B_M_LAST, 1UL));
	  B_TRY(BestMasterAttrPropSet(handle, B_M_DOLOOP, 1UL));

	  /* we MUST set the repeat ctr for all but a 25 */
	  if(!fIs2925)
	  {
	    B_TRY(BestMasterAttrPropSet(handle, B_M_REPEAT, NumPageRepeats));
	  }
	}
	else if (pProt->NumWaits != 0)
	{
	  /* all EXCEPT for the last phase ... put NumWaits as required */
	  B_TRY(BestMasterAttrPropSet(handle, B_M_WAITS, pProt->NumWaits));
	}
      
	/* This is to overcome the PCI bus bug in some PC's which will hang the PC */
	if (!fIs2925)
	{
	  B_TRY(BestMasterAttrPropSet(handle, B_M_RELREQ, 1UL));
	}
	else
	{
	  B_TRY(BestMasterAttrPropSet(handle, B_M_RELREQ, B_RELREQ_ON));
	}
      
	/* CONSIDER; try 64 bit access ?? */
	/* B_TRY(BestMasterAttrPropSet(handle, B_M_REQ64, 1)); */

	B_TRY(BestMasterAttrPhaseProg(handle));
      }


      /* the master block page entries get programmed onboard (2925 ONLY) */

      if (fIs2925)
      {
	SetupCmd = B_TSTSETUP_WRITEATTR;

	/* copy the setup for the phases on this attribute page to the card. */
	for (i = pProt->StartEntryNum; i <= pProt->EndEntryNum; i++)
	{
	  PBuffer[0] = i;
	  PBuffer[1] = AttrPageNum;
	  PBuffer[2] = NumTransfers;
	  PBuffer[3] = WriteCmd;
	  PBuffer[4] = pProt->ReadCmd;

	  B_TRY(BestBasicBlockWrite(handle, TEST_S_BUFFER, (b_int8ptr ) PBuffer, 20UL));
	  B_TRY(BestBasicBlockWrite(handle, TEST_SETUP, &SetupCmd, 1UL));
	}
      }
    }

  B_ERRETURN( B_TRY_RET);
}



/* --------------------------------------------------------------------------
 * CLI functions
 * -------------------------------------------------------------------------- */


/* --------------------------------------------------------------------------
 * Setup the analyser to behave as a PCI protocol error checker
 * DOCUMENTED
 * -------------------------------------------------------------------------- */

b_errtype EXPORT BestTestProtErrDetect(b_handletype handle)
{
  B_TRY_VARS_NO_PROG;
  b_errtype err;
  B_LICENSECHECK(B_CAPABILITY_ANALYZER)

  B_TRY_BEGIN
  {
    /* stop the analyzer and clear the observer status register */
    B_TRY(BestAnalyzerStop(handle));
    B_TRY(BestObsStatusClear(handle));

    /* clear the protocol-error-bit in the status register */
    B_TRY(BestStatusRegClear(handle, 0x10UL));

    /* setup the LA */
    B_TRY(BestTracePropSet(handle, B_TRC_HEARTBEATMODE, (b_int32)B_HBMODE_OFF));
    B_TRY(BestTracePattPropSet(handle, B_PT_SQ, "1"));
    B_TRY(BestTracePattPropSet(handle, B_PT_TRIGGER, "BERR"));

    /* restart the analyzer */
    B_TRY(BestAnalyzerRun(handle));
  }
  B_ERRETURN( B_TRY_RET);
}


/* --------------------------------------------------------------------------
 * DOCUMENTED
 * NOTE; The 2926A will return an abbreviated report until
 * the ASIC "error context capture" functions have been built into the CAPI.
 * -------------------------------------------------------------------------- */

b_errtype EXPORT BestTestResultDump(
    b_handletype handle,
    b_charptrtype filename
)
{
  B_DECLARE_FUNCNAME("BestTestResultDump [testrdump]");

  b_errtype err;
  ENUM_ERR_TEST testerr = ERR_TEST_NONE;

  FILE *fp_rpt = NULL;

  b_charptrtype pRptFileBuf;
  b_charptrtype pLaFileBuf;
  b_charptrtype char_ptr0;
  b_charptrtype char_ptr1;
  b_charptrtype char_ptr2;
  b_charptrtype pProdNumBuf = NULL;
  b_charptrtype pSerNumBuf = NULL;

  b_int8ptr buf = NULL;
  b_int8ptr pMasterBuf = NULL;
  b_int8ptr pCopyBuf = NULL;

  __int64 i;
  int n, m;
  b_int32 actual_timer, test_timer, trace_timer, obs_timer;
  b_int32 best_status, obs_status, la_status;
  b_int16 test_message = 0;
  b_int16 test_buf_length = 0;
  b_int32 first_error, accum_error;
  b_int32 addr1, addr2, blk_length;
  b_int32 actual_pos, mdw[4], cdw[4];
  b_int8 command;

  B_LICENSECHECK(B_CAPABILITY_ANALYZER)

  B_FCT_PARAM_NULL_POINTER_CHK (filename);

  /* generate the file names */
  i = strlen(filename);
  B_FCT_PARAM_CHK_R(2, (i == 0), "No filename passed");

  /* init timer values */
  test_timer = trace_timer = obs_timer = 0;

  if (NULL == (pRptFileBuf = malloc(i + 5)))
  {
    B_ERRETURN(B_E_HOST_MEM_FULL);
  }

  /*******************************************************/
  /* no returns after this line...goto clean_up_and_exit */

  if (NULL == (pLaFileBuf = malloc((size_t) (i + 5))))
  {
    err = B_E_HOST_MEM_FULL;
    goto clean_up_and_exit;
  }

  strcpy(pRptFileBuf, filename);
  strcpy(pLaFileBuf, filename);

  i--;
  char_ptr0 = filename + (int) i;
  char_ptr1 = pRptFileBuf + (int) i;
  char_ptr2 = pLaFileBuf + (int) i;

  m = 1;

  for (n = (int) i; n > 0; n--)
  {
    if (*char_ptr0 == '.')
    {
      m = 0;
      break;
    }
    if ((*char_ptr0 == '\\') || (*char_ptr0 == '/'))
    {
      break;
    }
    char_ptr0--;
    char_ptr1--;
    char_ptr2--;
  }

  if (m)
  {
    char_ptr1 = pRptFileBuf + (int) (i + 1);
    char_ptr2 = pLaFileBuf + (int) (i + 1);
  }

  strcpy(char_ptr1, ".rpt");
  strcpy(char_ptr2, ".wfm");

  /* open the report file */
  if (NULL == (fp_rpt = fopen(pRptFileBuf, "w")))
  {
    err = B_E_FILE_OPEN;
    goto clean_up_and_exit;
  }
  rewind(fp_rpt);

  /* write title */
  fprintf(fp_rpt, "Best Test Report File \n");
  fprintf(fp_rpt, "===================== \n\n");
  fprintf(fp_rpt, "Filename: %s\n", pRptFileBuf);

  FREE_N_NULL(pRptFileBuf);

  if ((err = BestVersionGet(handle, B_VER_PRODUCT, &pProdNumBuf)) != B_E_OK)
  {
    testerr = ERR_TEST_PRODNUM;
    goto clean_up_and_exit;
  }

  fprintf(fp_rpt, "Product Number: %s \n", pProdNumBuf);

  if ((err = BestVersionGet(handle, B_VER_SERIAL, &pSerNumBuf)) != B_E_OK)
  {
    testerr = ERR_TEST_SERIALNUM;
    goto clean_up_and_exit;
  }

  fprintf(fp_rpt, "Serial Number:  %s \n\n", pSerNumBuf);

  /* NOTE; 2926 does not yet support timing */

  if(BestIs2925(handle))
  {
    /* write time */
    if ((err = BestTestTimerValueGet(handle, B_TIM_ACTUALVAL, &actual_timer)) != B_E_OK)
    {
      testerr = ERR_TEST_TIMERVAL;
      goto clean_up_and_exit;
    }

    /* get test timers */
    if (actual_timer != 0)
    {
      ConvertTime(actual_timer, &hms);
      fprintf(fp_rpt, "Test Time: %02d:%02d:%02d \n\n", hms.hour, hms.minute, hms.sec);

      if ((err = BestTestTimerValueGet(handle, B_TIM_TESTERROR, &test_timer)) != B_E_OK
        || (err = BestTestTimerValueGet(handle, B_TIM_PROTERROR, &obs_timer)) != B_E_OK
        || (err = BestTestTimerValueGet(handle, B_TIM_TRACETRIG, &trace_timer)) != B_E_OK
        )
      {
        testerr = ERR_TEST_TIMERVAL;
        goto clean_up_and_exit;
      }
    }
  }

  /* get status information */
  testerr = ERR_TEST_STATUS;

  if ((err = BestStatusRegGet(handle, &best_status)) != B_E_OK
    || (err = BestObsStatusGet(handle, B_OBS_OBSSTAT, &obs_status)) != B_E_OK
    || (err = BestTraceStatusGet(handle, B_TRC_STAT, &la_status)) != B_E_OK
    )
  {
    goto clean_up_and_exit;
  }

  /* no test buffer in 2926 */
  if (BestIs2925(handle))
  {
    if ((err = BestBasicBlockRead(handle, TEST_BUFFER_LENGTH, (b_int8ptr )&i,
        4UL)) != B_E_OK)
      goto clean_up_and_exit;

    test_buf_length = (b_int16) (i & 0xffff);
    test_message = (b_int16)((b_int32)i >> 16);
  }

  testerr = ERR_TEST_NONE;

  /* write status information */
  fprintf(fp_rpt, "BOARD STATUS:\n");
  if (best_status & STATUS_MASTER_RUNNING)          /* master status */
  {
    fprintf(fp_rpt, "    Master:   running\n");
  }
  else
  {
    fprintf(fp_rpt, "    Master:   stopped\n");
  }

  if (best_status & STATUS_TARGET_RUNNING)          /* target status */
  {
    fprintf(fp_rpt, "    Target:   running\n");
  }
  else
  {
    fprintf(fp_rpt, "    Target:   stopped\n");
  }

  if (obs_status & OBS_STATUS_RUNNING)           /* observer status */
  {
    fprintf(fp_rpt, "    Observer: running");
    if (obs_status & OBS_STATUS_OUTOFSYNC)
    {
      fprintf(fp_rpt, " - out of sync\n");
    }
    else
    {
      fprintf(fp_rpt, "\n");
    }
  }
  else
  {
    fprintf(fp_rpt, "    Observer: stopped\n");
  }

  if (best_status & STATUS_LA_RUNNING)          /* la status */
  {
    fprintf(fp_rpt, "    LA:       running");
    if (la_status & LA_STATUS_TRIGGERED)
    {
      fprintf(fp_rpt, " - triggered\n\n");
    }
    else
    {
      fprintf(fp_rpt, "\n\n");
    }
  }
  else
  {
    fprintf(fp_rpt, "    LA:       stopped");
    if (la_status & LA_STATUS_TRIGGERED)
    {
      fprintf(fp_rpt, " / triggered\n\n");
    }
    else
    {
      fprintf(fp_rpt, "\n\n");
    }
  }

  fprintf(fp_rpt, "occured Errors: ");

  if ((best_status & STATUS_ANY_ERROR) == 0)
  {
    fprintf(fp_rpt, "none\n\n");
  }
  else
  {
    if (best_status & STATUS_DATA_COMPARE_ERR)
      fprintf(fp_rpt, "\n   Data Compare Error");

    if (best_status & STATUS_MASTER_BLOCK_ABORT)
      fprintf(fp_rpt, "\n   Master Block aborted");

    /* NOTE; different for 2925/2926 */
    if (HasTestFailed(handle, best_status))
      fprintf(fp_rpt, "\n   BestTestRun failed");

    fprintf(fp_rpt, "\n\n");
  }

  /* LA report */
  if (la_status & LA_STATUS_TRIGGERED)            /* when la trigger occured */
  {
    if (trace_timer != 0)  /* will only be non-0 for a 2925 */
    {
      ConvertTime(trace_timer, &hms);
      fprintf(fp_rpt, "LA trigger time: ");
      fprintf(fp_rpt, "%02d:%02d:%02d \n", hms.hour, hms.minute, hms.sec);
    }

    if ((err = BestTraceStop(handle)) != B_E_OK
      || (err = BestTraceMemoryFileDump(handle, pLaFileBuf)) != B_E_OK
      )
    {
      testerr = ERR_TEST_TRACEWRITE;
      goto clean_up_and_exit;
    }

    fprintf(fp_rpt, "LA Trace stored to file: %s\n\n", pLaFileBuf);
    FREE_N_NULL(pLaFileBuf);
  }


  /* Observer report */
  if (obs_status & OBS_STATUS_PROTERR)           /* when protocol error occured */
  {
    if ((err = BestObsStatusGet(handle, B_OBS_FIRSTERR, &first_error)) != B_E_OK
      || (err = BestObsStatusGet(handle, B_OBS_ACCUERR, &accum_error)) != B_E_OK
      )
    {
      testerr = ERR_TEST_OBSRESULT;
      goto clean_up_and_exit;
    }

    fprintf(fp_rpt, "detected protocol Errors:\n");
    fprintf(fp_rpt, "=========================\n");

    if (obs_timer != 0)  /* will only be non-0 for a 2925 */
    {
      ConvertTime(obs_timer, &hms);
      fprintf(fp_rpt, "first Error(s): - detection time: ");
      fprintf(fp_rpt, "%02d:%02d:%02d \n", hms.hour, hms.minute, hms.sec);
    }
    else
    {
      fprintf(fp_rpt, "first Error(s):\n");
    }

    /* TODO; 2926; extended error capability not yet implemented here */
    ReportObsErrors(handle, fp_rpt, first_error);

    fprintf(fp_rpt, "\nacummulated Error(s):\n");
    ReportObsErrors(handle, fp_rpt, accum_error);

    fprintf(fp_rpt, "\n\n");

  }
  else
    fprintf(fp_rpt, "no protocol Errors detected \n\n");

  /* test failed?..this is a TODO for 2926 when context reports are implemented */

  if (BestIs2925(handle) && 
      HasTestFailed(handle, best_status) && 
      (test_message == B_TSTMSG_RUN_ERROR))
  {
    /* alloc a minimum buffer length of 13 bytes */
    if (NULL == (buf = malloc(max(13, test_buf_length))))
    {
      testerr = ERR_TEST_MALLOC;
      err = B_E_HOST_MEM_FULL;
      goto clean_up_and_exit;
    }

    if ((err = BestTestBufferDump(handle, (b_int32 *) & i, buf)) != B_E_OK)
    {
      testerr = ERR_TEST_RESULTS_LOAD;
      goto clean_up_and_exit;
    }

    command = buf[0];

    addr1 = ((b_int32)buf[1]) |
            ((b_int32)buf[2] << 8) |
            ((b_int32)buf[3] << 16) |
            ((b_int32)buf[4] << 24);
    addr2 = ((b_int32)buf[5]) |
            ((b_int32)buf[6] << 8) |
            ((b_int32)buf[7] << 16) |
            ((b_int32)buf[8] << 24);
    blk_length = ((b_int32)buf[9]) |
                  ((b_int32)buf[10] << 8) |
                  ((b_int32)buf[11] << 16) |
                  ((b_int32)buf[12] << 24);

    FREE_N_NULL(buf);

    fprintf(fp_rpt, "Test function report\n");
    fprintf(fp_rpt, "====================\n");

    if (test_timer != 0)  /* will only be non-0 for a 2925 */
    {
      ConvertTime(test_timer, &hms);
      fprintf(fp_rpt, "   test failed at : ");
      fprintf(fp_rpt, "%02d:%02d:%02d \n", hms.hour, hms.minute, hms.sec);
    }

    switch (command & 0x3)
    {
    case B_TSTCMD_BLOCKMOVE:
    case B_TSTCMD_BLOCKMOVE_C:
      fprintf(fp_rpt, "   Test failed while copying data from\n");
      fprintf(fp_rpt, "    Source address:      %#8lX to\n", addr1);
      fprintf(fp_rpt, "    Destination address: %#8lX \n", addr2);
      fprintf(fp_rpt, "    Block length : %ld Bytes\n\n", blk_length << 2);
      break;

    case B_TSTCMD_TRAFFICMAKE:
    case B_TSTCMD_WRITEREAD:
    case B_TSTCMD_READ:
      fprintf(fp_rpt, "   Test failed while accessing\n");
      fprintf(fp_rpt, "    Start address: %#8lX \n", addr2);
      fprintf(fp_rpt, "    Block length:  %ld Bytes\n\n", blk_length << 2);
      break;

    default:
      assert(Abort && "Unanticipated case (switch (command & 0x3) in BestTestResultDump())");
    }

    /* report compare errors */
    if (best_status & STATUS_DATA_COMPARE_ERR)
    {
      if (NULL == (pMasterBuf = malloc(HOSTBUFFER))
        || NULL == (pCopyBuf = malloc(HOSTBUFFER))
        )
      {
        testerr = ERR_TEST_MALLOC;
        err = B_E_HOST_MEM_FULL;
        goto clean_up_and_exit;
      }

      fprintf(fp_rpt, "  -- compare Error -- memory dump\n");
/*		                             11111111112222222222333
         12345678:12345678901234567890123456789012 */
      fprintf(fp_rpt, " offset :           sent data            |");
      fprintf(fp_rpt, "         received data\n");
      actual_pos = 0;

      for (i = 0; (b_int64) i < ((long)blk_length * 4); i += HOSTBUFFER)
      {
        if ((err = BestHostIntMemDump(handle,
              UPPER_HALF_ADDR + (b_int32)i, (b_int32)HOSTBUFFER,
              (b_int8ptr) pMasterBuf)) != B_E_OK
          || (err = BestHostIntMemDump(handle, (b_int32) i, 
                    (b_int32)HOSTBUFFER, (b_int8ptr) pCopyBuf)) != B_E_OK
          )
        {
          testerr = ERR_TEST_RESULTS_LOAD;
          goto clean_up_and_exit;
        }

        /* go thru buffer in 4 DWORDS steps */
        for (n = 0; n < HOSTBUFFER; n += 16)
        {
          char_ptr1 = " ";

          for (m = 0; m < 4; m++)
          {
            actual_pos++;       /* actual DWORD position */
            mdw[m] = cdw[m] = 0;

            if (actual_pos <= blk_length) /*lint -e661 -e662 */
            {
              int offset = (n + m * 4);
              (void) BestStream2Long(&mdw[m], &pMasterBuf[offset], 1UL);
              (void) BestStream2Long(&cdw[m], &pCopyBuf[offset], 1UL);
            }

            if (mdw[m] != cdw[m])
            {
              char_ptr1 = "|";
            }
          }

          fprintf(fp_rpt, "%8lX:%8lX%8lX%8lX%8lX%s%8lX%8lX%8lX%8lX\n",
            (actual_pos << 2) - 16,
            mdw[0], mdw[1], mdw[2], mdw[3],
            char_ptr1,
            cdw[0], cdw[1], cdw[2], cdw[3]);

          if (actual_pos > blk_length)
            break;
        }
      }
      FREE_N_NULL(pMasterBuf);
      FREE_N_NULL(pCopyBuf);
    }
  }

  /* test is running? */
  if (BestIs2925(handle) && 
      (best_status & STATUS_MASTER_RUNNING) && 
      (test_message == B_TSTMSG_RUNNING))
  {
    /* alloc a minimum buffer length of 9 bytes */
    if (NULL == (buf = malloc(max(9, test_buf_length))))
    {
      testerr = ERR_TEST_MALLOC;
      err = B_E_HOST_MEM_FULL;
      goto clean_up_and_exit;
    }

    if ((err = BestTestBufferDump(handle, (b_int32 *) & i, buf)) != B_E_OK)
    {
      testerr = ERR_TEST_RESULTS_LOAD;
      goto clean_up_and_exit;
    }

    fprintf(fp_rpt, "Test function report\n");
    fprintf(fp_rpt, "====================\n");
    fprintf(fp_rpt, "   Test is running ... \n");

    if (test_buf_length > 1)
    {
      command = buf[0];

      addr1 = ((b_int32)buf[1]) |
              ((b_int32)buf[2] << 8) |
              ((b_int32)buf[3] << 16) |
              ((b_int32)buf[4] << 24);
      addr2 = ((b_int32)buf[5]) |
              ((b_int32)buf[6] << 8) |
              ((b_int32)buf[7] << 16) |
              ((b_int32)buf[8] << 24);

      switch (command & 0x3)
      {
      case B_TSTCMD_BLOCKMOVE:
      case B_TSTCMD_BLOCKMOVE_C:
        fprintf(fp_rpt, "   Copying data from\n");
        fprintf(fp_rpt, "    Source address:      %#8lX to\n", addr1);
        fprintf(fp_rpt, "    Destination address: %#8lX \n", addr2);
        break;

      case B_TSTCMD_TRAFFICMAKE:
      case B_TSTCMD_WRITEREAD:
      case B_TSTCMD_READ:
        fprintf(fp_rpt, "   Accessing\n");
        fprintf(fp_rpt, "    Start address: %#8lX \n", addr2);
        break;

      default:
        assert(Abort && "Unanticipated case (switch (command & 0x3) in BestTestResultDump())");
      }
    }
  }

clean_up_and_exit:              /* single exit takes care of all free/fclose */

  if (fp_rpt)
  {
    if (ERR_TEST_NONE != testerr)
      ReportError(testerr, fp_rpt);
    fclose(fp_rpt);
  }

  free(pRptFileBuf);
  free(pLaFileBuf);
  free(pMasterBuf);
  free(buf);
  free(pCopyBuf);

  B_ERRETURN( err);
}


/* --------------------------------------------------------------------------
 * Setting test properties
 * -------------------------------------------------------------------------- */


/* --------------------------------------------------------------------------
 * DOCUMENTED
 * -------------------------------------------------------------------------- */

b_errtype EXPORT BestTestPropSet(
    b_handletype handle,
    b_testproptype testprop,
    b_int32 value
)
{
  B_DECLARE_FUNCNAME("BestTestPropSet [testprpset]");
  b_errtype err;
  B_LICENSECHECK(B_CAPABILITY_ANALYZER)  /* includes handle check */

  switch (testprop)
  {
  case B_TST_BANDWIDTH:
    B_FCT_PARAM_CHK(3, value > 100);
    /* SCR; 24.3.98; limited to 90% */
    bit_props[handle].TestBandwidth = (b_int8) value; /* min(90, (b_int8) value); */
    break;

  case B_TST_BLKLENGTH:
    B_FCT_PARAM_CHK(3, (value < 1) || (value > (64UL * 1024UL)));
    B_FCT_PARAM_CHK(3, 0 != (value % 4));
    bit_props[handle].TestBlkLength = value;
    break;

  case B_TST_DATAPATTERN:
    B_FCT_PARAM_CHK(3, (value != B_DATAPATTERN_RANDOM)
      && (value != B_DATAPATTERN_FIX) && (value != B_DATAPATTERN_TOGGLE));
    bit_props[handle].TestDataPattern = (b_int8) value;
    break;

  case B_TST_PROTOCOL:
    B_FCT_PARAM_CHK(3, (value != B_PROTOCOL_LITE)
      && (value != B_PROTOCOL_MEDIUM) && (value != B_PROTOCOL_HARD));
    bit_props[handle].TestProtocol = (b_int8) value;
    break;

  case B_TST_COMPARE:
    B_FCT_PARAM_CHK(3, value > 1);
    bit_props[handle].TestCompare = (b_int8) value;
    break;

  case B_TST_STARTADDR:
    B_FCT_PARAM_CHK(3, 0 != (value % 4));
    bit_props[handle].TestStartAddr = value;
    break;

  case B_TST_NOFBYTES:
    B_FCT_PARAM_CHK(3, 0 != (value % 4));
    bit_props[handle].TestNofBytes = value;
    break;

  case B_TST_SOURCEADDR:
    B_FCT_PARAM_CHK(3, 0 != (value % 4));
    bit_props[handle].TestSourceAddr = value;
    break;

  case B_TST_DESTINADDR:
    B_FCT_PARAM_CHK(3, 0 != (value % 4));
    bit_props[handle].TestDestAddr = value;
    break;

  case B_TST_TIMEDIST:
    bit_props[handle].TestTimeDist = value;
    break;

  case B_TST_TIMELIMIT:
    bit_props[handle].TestTimeLimit = value;
    break;

  default:
    B_FCT_PARAM_ERROR(2, "Invalid Property");
  }
  B_ERRETURN( B_E_OK);
}


/* --------------------------------------------------------------------------
 * DOCUMENTED
 * -------------------------------------------------------------------------- */

b_errtype EXPORT BestTestPropDefaultSet(
    b_handletype handle
)
{
  b_errtype err;
  B_LICENSECHECK(B_CAPABILITY_ANALYZER)
  bit_props[handle] = def_bitprops;
  B_ERRETURN( B_E_OK);
}


/* --------------------------------------------------------------------------
 * BestTestRun(); The main man...running the test
 * DOCUMENTED
 * -------------------------------------------------------------------------- */

b_errtype EXPORT BestTestRun(
    b_handletype handle,
    b_int32 TestCmd
)
{
  B_DECLARE_FUNCNAME("BestTestRun [testrun]");

  b_int32 bus_addr1 = 0;
  b_int32 bus_addr2 = 0;
  b_int32 DecLocation;
  b_int32 NoOfTransfers, CacheLine, NoOfDwordsTotal;
  b_int32 NoOfBytesTotal;
  b_int32 NumCompleteFills;
  b_int32 SizeOfLastFill;
  b_int32 IntAddr;
  b_int32 size;
  b_int32 PBuffer[6];
  b_int32 value;
  b_int32 i;
  b_int32 AttrPageNum, AttrPageLen;
  b_int32 CacheLineTemp;
  int j;
  int NumAttrPages;

  b_int8 cmd;
  b_int8 SetupCmd;

  const B_PAP * pProt;

  int fIs2925 = BestIs2925(handle);
  b_errtype err;

  /* variables used after a catch must be declared volatile */
  volatile b_int8ptr  data_buffer = NULL;

  B_TRY_VARS_NO_PROG;
  B_LICENSECHECK(B_CAPABILITY_ANALYZER)

  /* interrupt command not allowed */
  B_FCT_PARAM_CHK(2, TestCmd == B_TSTCMD_INTGENERATE);

  /* get out if we've got a bandwidth of 0 */
  if(0 == bit_props[handle].TestBandwidth)
    B_ERRETURN(B_E_OK);

  B_TRY_BEGIN
  {
    switch (TestCmd)
    {
    case B_TSTCMD_TRAFFICMAKE:
      /* get the base address of decoder one */
      B_TRY(BestTargetDecoderRead(handle, B_DEC_STANDARD_1));
      B_TRY(BestTargetDecoderPropGet(handle, B_DEC_BASEADDR, &bus_addr1));

      /* get the size of decoder one */
      /* get out if decoder one is turned off */
      B_TRY(BestTargetDecoderPropGet(handle, B_DEC_SIZE, &size));
      B_TRY_FAIL(size == 0 ? B_E_TEST_NO_DECODER : B_E_OK);

      /* or memory decoding not enabled */
      B_TRY(BestTargetGenPropGet(handle, B_TGEN_MEMSPACE, &value));
      B_TRY_FAIL(value == 0 ? B_E_TEST_NO_DECODER : B_E_OK);

      /* decoder size MUST be >= block size */
      B_TRY_FAIL(
        ((b_int32)(1UL << (int)size) < bit_props[handle].TestBlkLength) ?
        B_E_DEC_INVALID_SIZE : B_E_OK);

      /* limit size to 64k in case we're talking to a 2925 who's only got
       * 128K total. Note that numbytes = 2^size  (i.e. 2^16 = 65536) 
       * TestBlkLength already limited to 64K. 
       */
      if (size > 16)
      {
        size = 16;
        B_TRY(BestTargetDecoderPropSet(handle, B_DEC_SIZE, size));
        B_TRY(BestTargetDecoderProg(handle, B_DEC_STANDARD_1));
      }

      /* all other decoders must be set for I/O or off ... first we need
       * to see if #2 is being used for top-half of 64-bit addr.
       */
      B_TRY(BestTargetDecoderPropGet(handle, B_DEC_LOCATION, &DecLocation));

      B_TRY(StopAllDecoders(handle, 
        (B_LOC_SPACE64 == DecLocation) ? B_DEC_STANDARD_3 : B_DEC_STANDARD_2));

      /* select target attr page 0 */
      B_TRY(BestTargetAttrPageSelect(handle, 0UL));

      /* source and destination are the same */
      bus_addr2 = bus_addr1;

      /* translate decoder size (exponent) into number of DWords */
      NoOfTransfers = (1UL << (int) (size - 2));

      /* don't use cache */
      CacheLine = 0;

      break;

    case B_TSTCMD_BLOCKMOVE:
      /* get the tsart addresses */
      bus_addr1 = bit_props[handle].TestSourceAddr;
      bus_addr2 = bit_props[handle].TestDestAddr;

    case B_TSTCMD_WRITEREAD:    /*lint !e616 deliberate flow into next case */
    case B_TSTCMD_READ:
      /* get the CacheLine size */
      B_TRY(BestConfRegGet(handle, 0x0cUL, &CacheLine));

      CacheLine &= 0xff;

      if (TestCmd == B_TSTCMD_BLOCKMOVE)
      {
        /* check cache size only for B_TST_BLOCKMOVE */
        /* can't use cache if the source and dest. addresses are not both
         * aligned to the cache line size (in DWords) */
        if ((CacheLine != 0)
          && (((bus_addr1 >> 2) % CacheLine) != ((bus_addr2 >> 2) % CacheLine)))
        {
          CacheLine = 0;
        }
      }
      else
      {
        /* for all but B_TST_BLOCKMOVE */
        bus_addr1 = bit_props[handle].TestStartAddr;
        bus_addr2 = bus_addr1;
      }

      /* get the number of transfers in DWords (at least 1) */
      NoOfTransfers = bit_props[handle].TestNofBytes;
      NoOfTransfers >>= 2;
      NoOfTransfers = max(1, NoOfTransfers);

      /* all decoders must be set for I/O or off */
      B_TRY(StopAllDecoders(handle, B_DEC_STANDARD_1));

      break;

    default:
      B_FCT_PARAM_ERROR(2, "Unknown Test Command");
    }

    /* get the # of bytes */
    NoOfBytesTotal = bit_props[handle].TestBlkLength;

    /* the block length is the number of DWords (at least 1) transferred for
     * the whole test. */
    NoOfDwordsTotal = NoOfBytesTotal >> 2;
    NoOfDwordsTotal = max(1, NoOfDwordsTotal);

    /* force both source and destination to be DWORD addresses */
    bus_addr1 &= 0xFFFFFFFCUL;
    bus_addr2 &= 0xFFFFFFFCUL;

    /* set and adjust the command if we're comparing */
    cmd = (b_int8) TestCmd;

    if (bit_props[handle].TestCompare)
    {
      if (TestCmd == B_TSTCMD_WRITEREAD)
      {
        cmd |= PBP_COMPARE;
      }
      else if (TestCmd == B_TSTCMD_BLOCKMOVE)
      {
        /* read data to lower half of internal mem AND compare */
        cmd |= (PBP_READ_TO_LOWER_HALF + PBP_COMPARE);
      }
    }

    /* load the card's internal memory */

    if ((TestCmd == B_TSTCMD_TRAFFICMAKE) || (TestCmd == B_TSTCMD_WRITEREAD))
    {
      /* allocate a buffer */
      data_buffer = (b_int8ptr) malloc((size_t)HOSTBUFFER);
      if(data_buffer == NULL)
        B_ERRETURN(B_E_HOST_MEM_FULL);

      /* and fill it */
      switch (bit_props[handle].TestDataPattern)
      {
      case B_DATAPATTERN_RANDOM:
        srand((unsigned int) time(NULL));
        for (i = 0; i < HOSTBUFFER; i++)
          *(data_buffer + (int) i) = (b_int8) rand();
        break;

      case B_DATAPATTERN_FIX:
        memset(data_buffer, 0, HOSTBUFFER);
        break;

      case B_DATAPATTERN_TOGGLE:
        for (i = 0; i < HOSTBUFFER; i += 8)
        {
          *((b_int32 *) (data_buffer + i)) = 0xa5a5a5a5UL;
          *((b_int32 *) (data_buffer + (i + 4))) = 0x5a5a5a5aUL;
        }
        break;

      default:
        B_FCT_PARAM_ERROR(2, "Unknown Test Data Pattern");
      }

      /* Load the buffer to card memory starting at UPPER_HALF_ADDR, in 1K
       * blocks. Note that we use the same 1K buffer as often as necessary ! */
      IntAddr = UPPER_HALF_ADDR;

      /* Calc and do the number of full loads (each HOSTBUFFER size) */

      NumCompleteFills = NoOfBytesTotal / HOSTBUFFER;

      for (i = 0; i < NumCompleteFills; i++)
      {
        B_TRY(BestHostIntMemFill(handle, IntAddr, (b_int32)HOSTBUFFER,
              (b_int8ptr)data_buffer));
        IntAddr += HOSTBUFFER;
      }

      /* and the last (partial) load ... last load must be on a dword
       * boundary...increase by 1 if necessary */

      if (0 != (SizeOfLastFill = ((NoOfBytesTotal + 3) % HOSTBUFFER) & ~0x3UL))
      {
        B_TRY(BestHostIntMemFill(handle, IntAddr, SizeOfLastFill, (b_int8ptr) data_buffer));
      }

      /* we're done with the buffer */
      FREE_N_NULL(data_buffer);
    }

    /* this only actually does something on a 2926 */
    B_TRY(ProgramBlockPage(handle, BLOCKPAGE_INIT, 0UL, 0UL, 0UL, 0UL, 0UL, 0UL,
          0UL));

    /* setup the master attribute pages */
    switch (bit_props[handle].TestProtocol)
    {
    case B_PROTOCOL_LITE:
      pProt = s_prot_lite;
      NumAttrPages = SIZEOF_S_PROT_LITE;
      CacheLineTemp = 1;
      break;

    case B_PROTOCOL_MEDIUM:
      pProt = s_prot_med;
      NumAttrPages = SIZEOF_S_PROT_MED;
      CacheLineTemp = CacheLine;
      break;

    case B_PROTOCOL_HARD:
      pProt = s_prot_hard;
      NumAttrPages = SIZEOF_S_PROT_HARD;
      CacheLineTemp = CacheLine;
      break;

    default:
      B_FCT_PARAM_ERROR(2, "Unknown Test Protocol Previously Set");
    }

    /* Program all phases in all attribute pages and each line in the block
     * page. The information needed to do that is in the static B_PAP arrays
     * above. */

    /* mem write & invalidate applicable only when the CacheLine size > 0 */
    fCanUseMWI = (CacheLineTemp != 0);

    if (CacheLineTemp == 0)
      CacheLineTemp = 4;

    for (j = 0, AttrPageNum = 1; j < NumAttrPages; j++)
    {
      B_TRY(ProgAttrPage(handle,
          &(pProt[j]),
          AttrPageNum,
          CacheLineTemp,        /* adjusted CacheLine */
          TestCmd,              /* command */
          &AttrPageLen));

      /* For the 2926 we add the Master Block entries for each attribute page
       * here (simply because we've got more necessary variables in scope here.
       * On the 2925 it's done in ProgAttrPage() 
       */
      if (!fIs2925)
      {
        /* Each Master Attr. Page gets one or more Block Page Entries */
        B_TRY(AddBlockPageEntries(handle, pProt, (b_int32)cmd, bus_addr1,
            bus_addr2, AttrPageNum, NoOfDwordsTotal));
      }

      /* increment the attribute page #, taking into account overflow
       * onto subsequent pages.... page size is 4 or 32 phases depending
       * on 2925 or 2926 but "auto-flows" onto the next page. */
      AttrPageNum += (1 + ((AttrPageLen - 1) / GetAttrPageSize(handle)));
    }

    if (!fIs2925)
    {
      /* Start the Master Run */
      B_TRY(BestMasterBlockPageRun(handle, 1UL));
    }
    else
    {
      /* setup and initialize the run (2925 ONLY) */
      PBuffer[0] = bus_addr1;
      PBuffer[1] = bus_addr2;
      PBuffer[2] = NoOfTransfers;
      PBuffer[3] = CacheLine;
      PBuffer[4] = NoOfDwordsTotal;
      PBuffer[5] = (b_int32) (int) rand();

      SetupCmd = B_TSTSETUP_INIT;

      /* start the test */
      B_TRY(BestBasicBlockWrite(handle, TEST_S_BUFFER, (b_int8ptr ) PBuffer, 24UL));
      B_TRY(BestBasicBlockWrite(handle, TEST_SETUP, &SetupCmd, 1UL));
      B_TRY(BestBasicBlockWrite(handle, TEST_RUN, &cmd, 1UL));
    }
  }

  B_TRY_CATCH
  {
    FREE_N_NULL(data_buffer);
  }

  B_ERRETURN(B_TRY_RET);
}


/* --------------------------------------------------------------------------
 * NOT DOCUMENTED
 * -------------------------------------------------------------------------- */

b_errtype EXPORT BestTestTimerRun(b_handletype handle)
{
  b_int8 cmd = B_TSTTIMER_RUN;
  b_errtype err;
  B_LICENSECHECK(B_CAPABILITY_ANALYZER)
  B_ERRETURN(BestBasicBlockWrite(handle, TEST_TIMER_CTRL, &cmd, 1UL));
}


/* --------------------------------------------------------------------------
 * NOT DOCUMENTED
 * -------------------------------------------------------------------------- */

b_errtype EXPORT BestTestTimerStop(b_handletype handle)
{
  b_int8 cmd = B_TSTTIMER_STOP;
  b_errtype err;
  B_LICENSECHECK(B_CAPABILITY_ANALYZER)
  B_ERRETURN(BestBasicBlockWrite(handle, TEST_TIMER_CTRL, &cmd, 1UL));
}


/* --------------------------------------------------------------------------
 * NOT DOCUMENTED
 * -------------------------------------------------------------------------- */

b_errtype EXPORT BestTestTimerValueGet(
    b_handletype handle,
    b_timertype timer,
    b_int32 * value)
{
  B_DECLARE_FUNCNAME("BestTestTimerValueGet [testtimvget]");

  b_int32 cmd;
  b_errtype err;
  B_LICENSECHECK(B_CAPABILITY_ANALYZER)
  B_FCT_PARAM_NULL_POINTER_CHK (value);

  switch (timer)
  {
  case B_TIM_ACTUALVAL:
    cmd = TEST_TIMER_ACTUAL;
    break;

  case B_TIM_TESTERROR:
    cmd = TEST_TIMER_TERROR;
    break;

  case B_TIM_PROTERROR:
    cmd = TEST_TIMER_PERROR;
    break;

  case B_TIM_TRACETRIG:
    cmd = TEST_TIMER_TTRIG;
    break;

  default:
    B_FCT_PARAM_ERROR(2, "Unknown Timer Type");
  }

  B_ERRETURN(BestBasicBlockRead(handle, cmd, (b_int8ptr ) value, 4UL));
}


/* --------------------------------------------------------------------------
 *
 * -------------------------------------------------------------------------- */

b_errtype EXPORT BestTestTimerRegClear(
    b_handletype handle,
    b_timertype timer
)
{
  B_DECLARE_FUNCNAME("BestTestTimerRegClear [testtimrclear]");
  b_int8 cmd;
  switch (timer)
  {
  case B_TIM_ACTUALVAL:
    cmd = B_TSTTIMER_CLEAR_ACTUAL;
    break;

  case B_TIM_TESTERROR:
    cmd = B_TSTTIMER_CLEAR_TERR;
    break;

  case B_TIM_PROTERROR:
    cmd = B_TSTTIMER_CLEAR_PERR;
    break;

  case B_TIM_TRACETRIG:
    cmd = B_TSTTIMER_CLEAR_TRACE;
    break;

  default:
    B_FCT_PARAM_ERROR(2, "Unknown Timer Type");
  }

  B_ERRETURN(BestBasicBlockWrite(handle, TEST_TIMER_CTRL, &cmd, 1UL));
}


/* --------------------------------------------------------------------------
 * NOT DOCUMENTED
 * -------------------------------------------------------------------------- */

b_errtype EXPORT BestTestBufferDump(
    b_handletype handle,
    b_int32 * num_of_bytes,
    b_int8ptr data_ptr)
{
  B_DECLARE_FUNCNAME("BestTestBufferDump [testbufdump]");

  b_errtype err;

  B_FCT_PARAM_NULL_POINTER_CHK (num_of_bytes);
  B_FCT_PARAM_NULL_POINTER_CHK (data_ptr);
  B_LICENSECHECK(B_CAPABILITY_ANALYZER)

  B_ERRCHECK(BestBasicBlockRead(handle, TEST_BUFFER_LENGTH,
            (b_int8ptr)num_of_bytes, 4UL));
  *num_of_bytes &= 0xFFFFUL;
  B_ERRETURN(BestBasicBlockRead(handle, TEST_BUFFER, data_ptr, *num_of_bytes));
}


/* --------------------------------------------------------------------------
 * NOT DOCUMENTED
 * -------------------------------------------------------------------------- */

b_errtype EXPORT BestTestBufferFill(
    b_handletype handle,
    b_int32 num_of_bytes,
    b_int8ptr data_ptr)
{
  B_DECLARE_FUNCNAME("BestTestBufferFill [testbuffill]");

  b_errtype err;
  
  B_FCT_PARAM_NULL_POINTER_CHK (data_ptr);
  B_FCT_PARAM_CHK(2, num_of_bytes > 1024);
  B_LICENSECHECK(B_CAPABILITY_ANALYZER)

  B_ERRCHECK(BestBasicBlockWrite(handle, TEST_BUFFER_LENGTH,
            (b_int8ptr )&num_of_bytes, 4UL));
  B_ERRETURN(BestBasicBlockWrite(handle, TEST_BUFFER, data_ptr, num_of_bytes));
}
